# MIT License

# Copyright (c) 2022 Intelligent Systems Lab Org

# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

# File author: Shariq Farooq Bhat

import torch
import torch.cuda.amp as amp
import torch.nn as nn

from zoedepth.trainers.loss import GradL1Loss, SILogLoss, FocalLossV1
from zoedepth.utils.config import DATASETS_CONFIG
from zoedepth.utils.misc import compute_metrics
from zoedepth.data.preprocess import get_black_border

from .base_trainer import BaseTrainer
from torchvision import transforms
from PIL import Image
import numpy as np

#"""
class DepthClassifier(nn.Module):
    def __init__(self, size, time):
        super().__init__()
        # size: adaptive pool size
        self.avg = torch.nn.AdaptiveAvgPool2d(size)
        self.avg1 = torch.nn.AdaptiveAvgPool2d((11,15))
        self.avg2 = torch.nn.AdaptiveAvgPool2d((22,30))
        self.avg3 = torch.nn.AdaptiveAvgPool2d((44,60))
        self.avg4 = torch.nn.AdaptiveAvgPool2d((88,120))
        self.avg5 = torch.nn.AdaptiveAvgPool2d((176,240))
        self.avg6 = torch.nn.AdaptiveAvgPool2d((352,480))

        self.refine_time = time
        self.up1 = torch.nn.Upsample(scale_factor=32, mode='bilinear',align_corners=True)
        self.up2 = torch.nn.Upsample(scale_factor=16, mode='bilinear',align_corners=True)
        self.up3 = torch.nn.Upsample(scale_factor=8, mode='bilinear',align_corners=True)
        self.up4 = torch.nn.Upsample(scale_factor=4, mode='bilinear',align_corners=True)
        self.up5 = torch.nn.Upsample(scale_factor=2, mode='bilinear',align_corners=True)
        self.up6 = torch.nn.Upsample(scale_factor=1, mode='bilinear',align_corners=True)

    def forward(self, depth):
        bs, c, h, w = depth.shape
        depth_with_blank = depth
        for t in range(self.refine_time):
            avg_method = [self.avg1, self.avg2, self.avg3, self.avg4, self.avg5, self.avg6][t]
            up_method = [self.up1, self.up2, self.up3, self.up4, self.up5, self.up6][t]
            for b in range(bs):
                single = depth[b, :, :, :]              # single1 480 640
                blank = depth_with_blank[b, :, :, :]==0 # blank:1 480 640
                avg = avg_method(single)                # avg   1 3 4
                avg = up_method(avg.unsqueeze(0)).squeeze(0)   #avg1 480 640
                single[blank] = avg[blank]
                depth[b, :, :, :] = single
        
        cls_seed = self.avg(depth)
        mask_near = torch.tensor(cls_seed < 20, dtype=torch.float32).to(depth.device) # 5.5 4.7

        return mask_near
    
#"""
class Trainer(BaseTrainer):
    def __init__(self, config, model, train_loader, test_loader=None, device=None):
        super().__init__(config, model, train_loader,
                         test_loader=test_loader, device=device)
        #map_size = [(11,15),(24,32),(48,64),(96,128),(192,256)]
        #self.map_name = ['refine1','refine2','refine3','refine4','refine5']
        self.device = device
        self.silog_loss = SILogLoss()
        self.focal_loss = FocalLossV1()
        #self.grad_loss = GradL1Loss()
        self.scaler = amp.GradScaler(enabled=self.config.use_amp)
        #self.drop_factor = [0.0 + i * 0.8/5 for i in range(5)] + [0.8 - i * 0.7/5 for i in range(5)]
        #self.map_size = map_size #bx1x12x16 bx1x24x32 bx1x48x64 bx1x96x128 bx1x192x256
        #self.depth_cls = DepthClassifier(size=(12,16), time=5)
        
        #"""
        self.focal_length = {
            'near':[0.001,  1, 0.25],
            'middle':[1.,  3.5,   0.25],
            'wide':[3.5,   5.5,   0.3],
            'ultra':[5.5,   10.0,    0.3]
        }
        #"""
        """
        self.focal_length = {
            'near':[0.001,  3.0, 0.125],
            'middle1':[3.0,  6.0,   0.125],
            'middle2':[6.0,  8.0,   0.125],
            'middle':[8.0,  15.0,   0.125],
            'wide1':[15.0,   20.0,   0.125],
            'wide2':[20.0,   30.0,   0.2],
            'wide':[30.0,   40.0,   0.2],
            'ultra':[40.0,   80.0,    0.2]
        }"""
    def train_on_batch(self, batch, train_step):
        """
        Expects a batch of images and depth as input
        batch["image"].shape : batch_size, c, h, w
        batch["depth"].shape : batch_size, 1, h, w
        """

        images, depths_gt = batch['image'].to(
            self.device), batch['depth'].to(self.device)
        dataset = batch['dataset'][0]

        b, c, h, w = images.size()
        mask = batch["mask"].to(self.device).to(torch.bool)
        # focal length mask
        Focal_mask = {k:None for k in self.focal_length}
        for focal in Focal_mask:
            Focal_mask[focal] = torch.logical_and(mask, torch.logical_and(depths_gt > self.focal_length[focal][0], depths_gt <= self.focal_length[focal][1]))
        Focal_mask['generic'] = mask
        # drop domain distance
        # compute drop ratio:
        """
        far_mask = depths_gt > 27 # 5.5 4.7
        near_mask = depths_gt < 5.0
        retain_mask = torch.logical_or(far_mask, near_mask)
        drop_mask = torch.rand(depths_gt.shape) > self.drop_factor[self.epoch]
        drop_mask = torch.logical_or(retain_mask, drop_mask.to(self.device))
        mask = torch.logical_and(drop_mask, mask)
        """
        losses = {}
        loss = 0.0

        #"""
        with amp.autocast(enabled=self.config.use_amp):
            output = self.model(images)
            focal_num = 0.
            for focal in Focal_mask:
                if not Focal_mask[focal].any():
                    continue
                else:
                    mask_flag = True
                    for m in Focal_mask[focal]:
                        if not m.any():# mask 全为false
                            mask_flag = False
                            break
                    if mask_flag:
                        pred_depths = output[focal]
                        l_si, pred = self.silog_loss(
                            pred_depths, depths_gt, mask=Focal_mask[focal], interpolate=True, return_interpolated=True)
                        loss = loss + l_si
                        focal_num = focal_num + 1.0
            loss = loss / focal_num
            losses[self.silog_loss.name] = loss

            if self.config.w_grad > 0:
                l_grad = self.grad_loss(pred, depths_gt, mask=mask)
                loss = loss + self.config.w_grad * l_grad
                losses[self.grad_loss.name] = l_grad
            else:
                l_grad = torch.Tensor([0])

        self.scaler.scale(loss).backward()

        self.scaler.step(self.optimizer)
        if self.should_log and (self.step % int(self.config.log_images_every * self.iters_per_epoch)) == 0:
            # -99 is treated as invalid depth in the log_images function and is colored grey.
            depths_gt[torch.logical_not(mask)] = -99

            self.log_images(rgb={"Input": images[0, ...]}, depth={"GT": depths_gt[0], "PredictedMono": pred[0]}, prefix="Train",
                            min_depth=DATASETS_CONFIG[dataset]['min_depth'], max_depth=DATASETS_CONFIG[dataset]['max_depth'])

            if self.config.get("log_rel", False):
                self.log_images(
                    scalar_field={"RelPred": output["relative_depth"][0]}, prefix="TrainRel") #feature={'feature_map':feature_map[0,0,:,:]},

            #self.log_images(
            #        scalar_field={"rel_depth": output["rel_depth"][0],"feature1": feat1,"feature2": feat2,"feature3": feat3,"feature4": feat4}, prefix="TrainRel") #feature={'feature_map':feature_map[0,0,:,:]},
            #self.log_images(
            #        scalar_field={"feature": feature_map[0,0,:,:]}, prefix="TrainFeat") #feature={'feature_map':feature_map[0,0,:,:]},
        self.scaler.update()
        self.optimizer.zero_grad()
        #"""
        return losses
    
    @torch.no_grad()
    def eval_infer(self, x):
        with amp.autocast(enabled=self.config.use_amp):
            m = self.model.module if self.config.multigpu else self.model
            preds = m(x)

        return preds['generic']

    @torch.no_grad()
    def crop_aware_infer(self, x):
        # if we are not avoiding the black border, we can just use the normal inference
        if not self.config.get("avoid_boundary", False):
            return self.eval_infer(x)
        
        # otherwise, we need to crop the image to avoid the black border
        # For now, this may be a bit slow due to converting to numpy and back
        # We assume no normalization is done on the input image

        # get the black border
        assert x.shape[0] == 1, "Only batch size 1 is supported for now"
        x_pil = transforms.ToPILImage()(x[0].cpu())
        x_np = np.array(x_pil, dtype=np.uint8)
        black_border_params = get_black_border(x_np)
        top, bottom, left, right = black_border_params.top, black_border_params.bottom, black_border_params.left, black_border_params.right
        x_np_cropped = x_np[top:bottom, left:right, :]
        x_cropped = transforms.ToTensor()(Image.fromarray(x_np_cropped))

        # run inference on the cropped image
        pred_depths_cropped = self.eval_infer(x_cropped.unsqueeze(0).to(self.device))

        # resize the prediction to x_np_cropped's size
        pred_depths_cropped = nn.functional.interpolate(
            pred_depths_cropped, size=(x_np_cropped.shape[0], x_np_cropped.shape[1]), mode="bilinear", align_corners=False)
        

        # pad the prediction back to the original size
        pred_depths = torch.zeros((1, 1, x_np.shape[0], x_np.shape[1]), device=pred_depths_cropped.device, dtype=pred_depths_cropped.dtype)
        pred_depths[:, :, top:bottom, left:right] = pred_depths_cropped

        return pred_depths



    def validate_on_batch(self, batch, val_step):
        input_RGB = batch['image'].to(self.device)
        depths_gt = batch['depth'].to(self.device)
        #print(depths_gt.max())
        #print(depths_gt.mean())
        dataset = batch['dataset'][0]
        mask = batch["mask"].to(self.device) # 1x1x352x1216
        if 'has_valid_depth' in batch:
            if not batch['has_valid_depth']:
                return None, None

        depths_gt = depths_gt.squeeze().unsqueeze(0).unsqueeze(0)
        mask = mask.squeeze().unsqueeze(0).unsqueeze(0)
        if dataset == 'nyu':
            pred_depths = self.crop_aware_infer(input_RGB)
        else:
            with torch.no_grad():
                if True:# slide
                    bs, _, h, w = input_RGB.shape
                    assert w > h and bs == 1
                    interval_all = w - 480 # 1216-352 = 864
                    shift_size = 3
                    interval = interval_all // (shift_size-1) # shift_size = 16  864//15=57
                    sliding_images = []
                    sliding_masks = torch.zeros((bs, 1, h, w), device=input_RGB.device) # 352x352
                    for i in range(shift_size):
                        sliding_images.append(input_RGB[..., :, i*interval:i*interval+480])
                        sliding_masks[..., :, i*interval:i*interval+480] += 1
                    input_RGB = torch.cat(sliding_images, dim=0)# 3x3x352x480
                #if False:# flip
                #    input_RGB = torch.cat((input_RGB, torch.flip(input_RGB, [3])), dim=0) # 32x3x352x352
                pred_s = torch.zeros((bs, 1, h, w), device=input_RGB.device)
                for i in range(3):
                    pred_d = self.eval_infer(input_RGB[i,:,:,:].unsqueeze(0))
                    pred_d = nn.functional.interpolate(pred_d, [352, 480], mode='bilinear', align_corners=True)
                    pred_s[..., :, i*interval:i*interval+480] += pred_d
                pred_depths = pred_s/sliding_masks # 352x1216
        if dataset == 'nyu':
            pred_depths = pred_depths.squeeze().unsqueeze(0).unsqueeze(0)
        else:
            pass#pred_d = nn.functional.interpolate(pred_d, [352, 480], mode='bilinear', align_corners=True)
        if False:#args.flip_test:
            batch_s = pred_d.shape[0]//2
            pred_d = (pred_d[:batch_s] + torch.flip(pred_d[batch_s:], [3]))/2.0
        """
        if dataset == 'kitti':#args.shift_window_test:
            pred_s = torch.zeros((bs, 1, h, w), device=pred_d.device)
            for i in range(shift_size):
                pred_s[..., :, i*interval:i*interval+480] += pred_d[i:i+1]
            pred_depths = pred_s/sliding_masks # 352x1216
        """
        with amp.autocast(enabled=self.config.use_amp):
            l_depth = self.silog_loss(
                pred_depths, depths_gt, mask=mask.to(torch.bool), interpolate=True)

        metrics = compute_metrics(depths_gt, pred_depths, **self.config)
        losses = {f"{self.silog_loss.name}": l_depth.item()}

        if val_step == 1 and self.should_log:
            depths_gt[torch.logical_not(mask)] = -99
            self.log_images(rgb={"Input": input_RGB[0]}, depth={"GT": depths_gt[0], "PredictedMono": pred_depths[0]}, prefix="Test",
                            min_depth=DATASETS_CONFIG[dataset]['min_depth'], max_depth=DATASETS_CONFIG[dataset]['max_depth'])

        return metrics, losses
